/*******************************************************************
*
*  DESCRIPTION: auditor
*
*  AUTHOR: Shagun Goel
*
*
**
*******************************************************************/

/** include files **/
#include "auditor.h"  // class auditor
#include "message.h"   	  // class ExternalMessage, InternalMessage
#include "mainsimu.h"     // MainSimulator::Instance().getParameter( ... )

/*******************************************************************
* Function Name: auditor
* Description: 	
* 		
********************************************************************/
auditor::auditor( const string &name )
: Atomic( name )
, enter_sys( addInputPort( "enter_sys" ) )
, exit_q( addInputPort( "exit_q" ) )
, exit_sys( addInputPort( "exit_sys" ) )
, in( addInputPort( "in" ) )
, exp_custsys( addOutputPort( "exp_custsys" ) )
, exp_custque( addOutputPort( "exp_custque" ) )
, wait_que( addOutputPort( "wait_que" ) )
, wait_sys( addOutputPort( "wait_sys" ) )
{
//empty	
}

/*******************************************************************
* Function Name: initFunction
* Description:
* 
********************************************************************/
Model& auditor::initFunction()
{
	
in_sys = 0;
out_q = 0;
out_sys = 0;
exp_custsys_val = 0;
wait_que_val = 0;
exp_custque_val = 0;
wait_sys_val = 0;
arrivalTimesexp_custque.erase( arrivalTimesexp_custque.begin(), arrivalTimesexp_custque.end() );
arrivalTimeswait_sys.erase( arrivalTimeswait_sys.begin(), arrivalTimeswait_sys.end() );

return *this ;
}

/*******************************************************************
* Function Name: externalFunction
* Description: 
********************************************************************/
Model& auditor::externalFunction( const ExternalMessage &msg )
{

if (msg.port() == enter_sys)
	{
	in_sys++;  // customer just enetered the system.
	arrivalTimesexp_custque.push_back(msg.time());  // hold customer arrival time.
	calculate_exp_custsys();
	calculate_wait_que();
	}
else if (msg.port() == exit_q)
	{
	out_q++;  // customer just left the queue.
	calculate_exp_custsys();
	calculate_exp_custque(msg.time());
	}
else if (msg.port() == exit_sys)
	{
	out_sys++;  // customer just left the system.
	calculate_wait_que();
	calculate_wait_sys(msg.time());
	}
else if (msg.port() == in)
	{
	holdIn( active, Time::Zero);
	}

return *this;
}

/*******************************************************************
* Function Name: internalFunction
* Description: 
********************************************************************/
Model& auditor::internalFunction( const InternalMessage & )
{
passivate();
return *this ;
}

/*******************************************************************
* Function Name: outputFunction
* Description: 
********************************************************************/
Model &auditor::outputFunction( const InternalMessage &msg )
{

sendOutput( msg.time(), exp_custsys, exp_custsys_val);
sendOutput( msg.time(), exp_custque, exp_custque_val);
sendOutput( msg.time(), wait_que, wait_que_val);
sendOutput( msg.time(), wait_sys, wait_sys_val);

return *this;
}
/************************************************
* The mean of elements in the queue
*************************************************/
void auditor::calculate_exp_custsys()
{
int q_update = in_sys + out_q;
int cur_q_size = in_sys - out_q;

exp_custsys_val = ((exp_custsys_val * q_update) + cur_q_size) / (q_update + 1);
}
/***********************************************
* The mean of elements in system
* *********************************************/
void auditor::calculate_wait_que()
{
int sys_update = in_sys + out_sys;
int cur_sys_size = in_sys - out_sys;

wait_que_val = ((wait_que_val * sys_update) + cur_sys_size) / (sys_update + 1);
}
/************************************************
* The mean of waiting time in the queue
*************************************************/
void auditor::calculate_exp_custque(const Time &cur_time)
{
Time cur_wating_time;
float cur_wating_time_sec;

MASSERT (!arrivalTimesexp_custque.empty());

if (cur_time >= arrivalTimesexp_custque.front() )
	{
	cur_wating_time = cur_time - arrivalTimesexp_custque.front();  // time diff
	cur_wating_time_sec = convertTimeToSec(cur_wating_time);

	MASSERT(out_q > 0);
	exp_custque_val = ( (exp_custque_val * (out_q - 1)) + cur_wating_time_sec ) / out_q;
	}
arrivalTimeswait_sys.push_back(arrivalTimesexp_custque.front());
arrivalTimesexp_custque.pop_front();
}
/************************************************
* The mean of waiting time in the system
*************************************************/
void auditor::calculate_wait_sys(const Time &cur_time)
{
Time cur_wating_time;
float cur_wating_time_sec;

MASSERT (!arrivalTimeswait_sys.empty());

if (cur_time >= arrivalTimeswait_sys.front() )
	{
	cur_wating_time = cur_time - arrivalTimeswait_sys.front();  // time diff
	cur_wating_time_sec = convertTimeToSec(cur_wating_time);

	MASSERT(out_sys > 0);
	wait_sys_val = ( (wait_sys_val * (out_sys - 1)) + cur_wating_time_sec ) / out_sys;
	}
arrivalTimeswait_sys.pop_front();
}
/************************************************
* Convert time into seconds
*************************************************/
float auditor::convertTimeToSec(const Time &time)
{
int hour = 60 * 60 * time.hours();
int min = 60 * time.minutes();
int sec = time.seconds();
float msec = (float)time.mseconds()/1000;

return hour + min + sec + msec;
}
